<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

$logFile = __DIR__ . "/../otp_debug.log";

// Detect calling service automatically
$callingService = "unknown";

// Lambda
if (getenv('AWS_LAMBDA_FUNCTION_NAME')) {
    $callingService = "Lambda";
}
// EC2 (check metadata endpoint)
elseif (@file_get_contents(
    'http://169.254.169.254/latest/meta-data/instance-id',
    false,
    stream_context_create(['http' => ['timeout' => 0.1]])
)) {
    $callingService = "EC2";
}
// Cognito
elseif (!empty($_SERVER['HTTP_X_AMZ_COGNITO_IDENTITY'])) {
    $callingService = "Cognito";
}

// Detect which script/API triggered this request
// Optionally, you can send a custom header "X-Calling-Script" from caller
$callingScript = $_SERVER['HTTP_X_CALLING_SCRIPT'] 
    ?? ($_SERVER['SCRIPT_FILENAME'] ?? 'unknown');

// Build log entry
$logEntry = [
    "timestamp"       => date("Y-m-d H:i:s"),
    "full_path"       => realpath(__FILE__),                        // Current script full path
    "source_ip"       => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
    "forwarded_for"   => $_SERVER['HTTP_X_FORWARDED_FOR'] ?? 'none', // Real client if behind LB
    "referer"         => $_SERVER['HTTP_REFERER'] ?? 'none',         // Caller page/script
    "user_agent"      => $_SERVER['HTTP_USER_AGENT'] ?? 'none',
    "request_method"  => $_SERVER['REQUEST_METHOD'] ?? 'none',
    "query_string"    => $_SERVER['QUERY_STRING'] ?? 'none',
    "post_data"       => $_POST ?: [],
    "calling_service" => $callingService,
    "calling_script"  => $callingScript
];

// Append log as JSON line
file_put_contents($logFile, json_encode($logEntry) . PHP_EOL, FILE_APPEND);


require_once __DIR__ . '/../aws_get_env.php'; // fetch AWS keys dynamically
require __DIR__ . '/../vendor/autoload.php';

use Aws\Sns\SnsClient;
use Aws\Exception\AwsException;

// Input validation and sanitization
$phonenumber = $_POST["username"] ?? null;
$otp         = $_POST["iveposOtp"] ?? random_int(100000, 999999);

if (!$phonenumber) {
    echo json_encode([
        "status" => "failed",
        "error"  => "Phone number missing"
    ]);
    exit;
}

// Blocked number codes
$blockedCodes = ['+880', '+964', '+20','+93'];
if (isBlockedNumber($phonenumber, $blockedCodes)) {
    echo json_encode([
        "status" => "failed",
        "error"  => "Blocked phone number"
    ]);
    exit;
}

// Fetch AWS credentials dynamically
$awsAccessKey = get_aws_secret("AWS-ACCESS-KEY");
$awsSecretKey = get_aws_secret("AWS-SECRET-KEY");

// SNS Client
$sns = new SnsClient([
    'credentials' => [
        'key'    => $awsAccessKey,
        'secret' => $awsSecretKey
    ],
    'region'  => 'ap-south-1',
    'version' => 'latest'
]);

$message = 'Your IVEPOS/Intuition OTP for secure login is : ' . $otp;

$args = [
    'Message'     => $message,
    'PhoneNumber' => $phonenumber
];

try {
    $result = $sns->publish($args);

    echo json_encode([
        "status"      => "success",
        "otp"         => $otp,
        "phoneNumber" => $phonenumber,
        "message"     => $result
    ]);

} catch (AwsException $e) {
    error_log($e->getMessage());

    echo json_encode([
        "status" => "failed",
        "error"  => $e->getAwsErrorMessage()
    ]);
}

// Function to check if phone number starts with any blocked code
function isBlockedNumber($phonenumber, $blockedCodes) {
    foreach ($blockedCodes as $code) {
        if (strpos($phonenumber, $code) === 0) {
            return true;
        }
    }
    return false;
}
?>
