<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header('Content-Type: application/json');

$servername = "ivepos-db-production-regular.mysql.database.azure.com";
$username   = "iveposadmin";
$password   = "Intuition123";
$licenseDB  = "ivepos_license_retail";

// ----------------------
// Fetch aggregated sums
// ----------------------
function getSalesSum($conn, $vendor_name, $vendor_id) {
    $sql = "
        SELECT 
            SUM(total) AS total_sum,
            SUM(CASE WHEN (vendor_payment_status IS NULL OR vendor_payment_status = '' OR vendor_payment_status <> 'success') 
                     THEN purchase_price_total ELSE 0 END) AS ppt_sum,
            SUM(CASE WHEN (profit_payment_status IS NULL OR profit_payment_status = '' OR profit_payment_status <> 'success') 
                     THEN profit ELSE 0 END) AS profit_sum
        FROM All_Sales
        WHERE vendor_name = ? 
          AND vendor_id = ? 
    ";

    $stmt = $conn->prepare($sql);
    if ($stmt === false) {
        die("SQL prepare failed in getSalesSum: " . $conn->error . " | Query: " . $sql);
    }

    $stmt->bind_param("ss", $vendor_name, $vendor_id);
    $stmt->execute();
    $stmt->bind_result($t, $ppt, $pft);
    $stmt->fetch();
    $stmt->close();

    return [
        'total' => (float) ($t ?? 0),
        'purchase_price_total' => (float) ($ppt ?? 0),
        'profit' => (float) ($pft ?? 0)
    ];
}

// ----------------------
// Fetch MAX(_id)
// ----------------------
function getMaxId($conn, $vendor_name, $vendor_id) {
    $sql = "
        SELECT MAX(_id) 
        FROM All_Sales 
        WHERE vendor_name = ? 
          AND vendor_id = ? 
          AND (
                (vendor_payment_status IS NULL OR vendor_payment_status = '' OR vendor_payment_status <> 'success')
                OR
                (profit_payment_status IS NULL OR profit_payment_status = '' OR profit_payment_status <> 'success')
              )
    ";
    $stmt = $conn->prepare($sql);
    if ($stmt === false) {
        die("SQL prepare failed in getMaxId: " . $conn->error . " | Query: " . $sql);
    }

    $stmt->bind_param("ss", $vendor_name, $vendor_id);
    $stmt->execute();
    $stmt->bind_result($max_id);
    $stmt->fetch();
    $stmt->close();

    return (int) ($max_id ?? 0);
}

// ----------------------
// Fetch bill numbers
// ----------------------
function getBillNos($conn, $vendor_name, $vendor_id) {
    $sql = "
        SELECT DISTINCT bill_no 
        FROM All_Sales 
        WHERE vendor_name = ? 
          AND vendor_id = ? 
          AND (
                (vendor_payment_status IS NULL OR vendor_payment_status = '' OR vendor_payment_status <> 'success')
                OR
                (profit_payment_status IS NULL OR profit_payment_status = '' OR profit_payment_status <> 'success')
              )
    ";
    $stmt = $conn->prepare($sql);
    if ($stmt === false) {
        die("SQL prepare failed in getBillNos: " . $conn->error . " | Query: " . $sql);
    }

    $stmt->bind_param("ss", $vendor_name, $vendor_id);
    $stmt->execute();
    $stmt->bind_result($billNo);
    $billNos = [];
    while ($stmt->fetch()) {
        $billNos[] = $billNo;
    }
    $stmt->close();

    return $billNos;
}

// ----------------------
// Main connection
// ----------------------
$mainConn = new mysqli($servername, $username, $password, $licenseDB);
if ($mainConn->connect_error) {
    echo json_encode(['status' => 'failed', 'message' => 'Main DB connection failed: ' . $mainConn->connect_error]);
    exit;
}

// Validate POST fields
$required = ['email','company','store','vendor_name','vendor_id'];
foreach ($required as $field) {
    if (!isset($_POST[$field])) {
        echo json_encode(['status' => 'failed', 'message' => "Missing parameter: $field"]);
        exit;
    }
}

$email       = $_POST['email'];
$company     = $_POST['company'];
$store       = $_POST['store'];
$vendor_name = $_POST['vendor_name'];
$vendor_id   = $_POST['vendor_id'];

// ----------------------
// Validate company
// ----------------------
$stmt = $mainConn->prepare("SELECT id FROM company WHERE email = ? AND company = ?");
if ($stmt === false) {
    die("SQL prepare failed (company): " . $mainConn->error);
}
$stmt->bind_param("ss", $email, $company);
$stmt->execute();
$stmt->bind_result($companyid);
if (!$stmt->fetch()) {
    echo json_encode(['status' => 'failed', 'message' => 'Company not found or unauthorized']);
    exit;
}
$stmt->close();

// ----------------------
// Validate store
// ----------------------
$stmt = $mainConn->prepare("SELECT id FROM store WHERE store = ?");
if ($stmt === false) {
    die("SQL prepare failed (store): " . $mainConn->error);
}
$stmt->bind_param("s", $store);
$stmt->execute();
$stmt->bind_result($storeid);
if (!$stmt->fetch()) {
    echo json_encode(['status' => 'failed', 'message' => 'Store not found']);
    exit;
}
$stmt->close();

// ----------------------
// Retrieve devices
// ----------------------
$stmt = $mainConn->prepare("SELECT device FROM device WHERE storeid = ?");
if ($stmt === false) {
    die("SQL prepare failed (device): " . $mainConn->error);
}
$stmt->bind_param("i", $storeid);
$stmt->execute();
$stmt->bind_result($deviceName);

$devices = [];
while ($stmt->fetch()) {
    $devices[] = $deviceName;
}
$stmt->close();

if (empty($devices)) {
    echo json_encode(['status' => 'failed', 'message' => 'No devices found for this store']);
    exit;
}

// ----------------------
// Find matching DBs
// ----------------------
$pattern = $company . "_" . $store . "_%_mydbsalesdata";
$stmt = $mainConn->prepare("SELECT SCHEMA_NAME FROM information_schema.SCHEMATA WHERE SCHEMA_NAME LIKE ?");
if ($stmt === false) {
    die("SQL prepare failed (schemas): " . $mainConn->error);
}
$stmt->bind_param("s", $pattern);
$stmt->execute();
$stmt->bind_result($dbName);

$response = [
    'status' => 'success',
    'total' => 0.0,
    'purchase_price_total' => 0.0,
    'profit' => 0.0,
    'databases' => new stdClass()
];

$successCount = 0;

while ($stmt->fetch()) {
    $conn = new mysqli($servername, $username, $password, $dbName);
    if ($conn->connect_error) {
        $response['databases']->$dbName = ['error' => 'DB connection failed: ' . $conn->connect_error];
        continue;
    }

    $sumMetrics = getSalesSum($conn, $vendor_name, $vendor_id);
    $response['total'] += $sumMetrics['total'];
    $response['purchase_price_total'] += $sumMetrics['purchase_price_total'];
    $response['profit'] += $sumMetrics['profit'];

    $response['databases']->$dbName = [
        'total' => $sumMetrics['total'],
        'purchase_price_total' => $sumMetrics['purchase_price_total'],
        'profit' => $sumMetrics['profit'],
        'max_id' => getMaxId($conn, $vendor_name, $vendor_id),
        'bill_nos' => getBillNos($conn, $vendor_name, $vendor_id)
    ];

    $successCount++;
    $conn->close();
}

$stmt->close();
$mainConn->close();

if ($successCount === 0) {
    $response['status'] = 'failed';
    $response['message'] = 'No valid databases found';
}

echo json_encode($response);
