<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

$config = require __DIR__ . '/../config.php';// loads DB info
require_once __DIR__ . '/../aws_get_env.php'; // fetch AWS keys dynamically
require __DIR__ . '/../vendor/autoload.php';

$mysqli = new mysqli(
    $config['db_host'],
    $config['db_user'],
    $config['db_pass'],
    $config['db_name']
);

if ($mysqli->connect_error) {
    die("Connection failed: " . $mysqli->connect_error);
}

// Helper: Base64URL decode
function base64url_decode($data) {
    $data = strtr($data, '-_', '+/');           // convert to standard base64
    $padding = strlen($data) % 4;
    if ($padding > 0) {
        $data .= str_repeat('=', 4 - $padding); // restore missing padding
    }
    return base64_decode($data);
}

$phone   = $_POST['phoneNumber'] ?? null;
$message = $_POST['message'] ?? null;
$token   = $_POST['auth_token'] ?? null;

if (!$phone || !$message || !$token) {
    echo json_encode(["error" => "Missing fields"]);
    exit;
}

// Decode token (Base64URL safe)
$decoded = base64url_decode($token);
if ($decoded === false || strlen($decoded) < 48) {
    echo json_encode(["error" => "Invalid token format"]);
    exit;
}

$ivlen = openssl_cipher_iv_length("AES-256-CBC");
$iv = substr($decoded, 0, $ivlen);
$hmac = substr($decoded, $ivlen, 32);
$ciphertext = substr($decoded, $ivlen + 32);

$calc_hmac = hash_hmac("sha256", $iv . $ciphertext, $config['secret_hmac'], true);
if (!hash_equals($hmac, $calc_hmac)) {
    echo json_encode(["error" => "Token integrity failed"]);
    exit;
}

// Decrypt payload
$payload = openssl_decrypt($ciphertext, "AES-256-CBC", $config['secret_key'], OPENSSL_RAW_DATA, $iv);
if ($payload === false) {
    echo json_encode(["error" => "Token decryption failed"]);
    exit;
}

// Extract expiry from payload
list(, $expiry) = explode("|", $payload);
if (time() > (int)$expiry) {
    echo json_encode(["error" => "Token expired"]);
    exit;
}

// ✅ Validate token in DB (only check if exists and not expired)
$stmt = $mysqli->prepare("SELECT id FROM tokens WHERE token=? AND expiry > NOW()");
$stmt->bind_param("s", $token);
$stmt->execute();
$stmt->store_result();
if ($stmt->num_rows === 0) {
    echo json_encode(["error" => "Token not found or expired"]);
    exit;
}
$stmt->close();

// ✅ Send SMS using AWS SNS


use Aws\Sns\SnsClient;
use Aws\Exception\AwsException;


// Fetch AWS credentials dynamically
$awsAccessKey = get_aws_secret("AWS-ACCESS-KEY");
$awsSecretKey = get_aws_secret("AWS-SECRET-KEY");

// SNS Client
$params = [
    'credentials' => [
        'key'    => $awsAccessKey,
        'secret' => $awsSecretKey
    ],
    'region'  => 'ap-south-1',
    'version' => 'latest'
];

$sns = new SnsClient($params);

// Prepare message

$args = [
    'Message'     => $message,
    'PhoneNumber' => $phone
];

$blockedCodes = ['+880', '+964', '+20','+93'];

try {
    if (isBlockedNumber($phone, $blockedCodes)) {
        echo json_encode([
            "status" => "failed",
            "message" => "SMS blocked for this country code"
        ]);
        exit;
    }

    $result = $sns->publish($args);

    echo json_encode([
        "status" => "success",
        "message" => $result
    ]);

} catch (AwsException $e) {
    error_log($e->getMessage());
    echo json_encode([
        "status" => "failed",
        "error" => $e->getAwsErrorMessage()
    ]);
}

// Function to check if phone number starts with any blocked code
function isBlockedNumber($phone, $blockedCodes) {
    foreach ($blockedCodes as $code) {
        if (strpos($phone, $code) === 0) {
            return true;
        }
    }
    return false;
}
?>
