<?php

/**
 * Firebase Cloud Messaging HTTP v1 API implementation
 * Upgraded from legacy FCM API to v1 API
 * @author Firebase FCM v1 Migration
 */
class Firebase {
    
    private $projectId;
    private $serviceAccountPath;
    private $accessToken;
    private $tokenExpiry;
    
    public function __construct($projectId, $serviceAccountPath) {
        $this->projectId = $projectId;
        $this->serviceAccountPath = $serviceAccountPath;
        $this->accessToken = null;
        $this->tokenExpiry = 0;
    }
    
    /**
     * Send push message to single user by firebase registration token
     */
    public function send($token, $message) {
        $payload = array(
            'message' => array(
                'token' => $token,
                'data' => $message,
            )
        );
        return $this->sendPushNotification($payload);
    }
    
    /**
     * Send message to a topic by topic name
     */
    public function sendToTopic($topic, $message) {
        $payload = array(
            'message' => array(
                'topic' => $topic, // Remove '/topics/' prefix as v1 API handles it
                'data' => $message,
            )
        );
        return $this->sendPushNotification($payload);
    }
    
    /**
     * Send push message to multiple users by firebase registration tokens
     * Note: v1 API doesn't support multiple tokens in single request
     * This method sends individual requests for each token
     */
    public function sendMultiple($registrationTokens, $message) {
        $results = array();
        
        foreach ($registrationTokens as $token) {
            $result = $this->send($token, $message);
            $results[] = array(
                'token' => $token,
                'result' => $result
            );
        }
        
        return $results;
    }
    
    /**
     * Send notification message (with title and body)
     */
    public function sendNotification($token, $title, $body, $data = array()) {
        $payload = array(
            'message' => array(
                'token' => $token,
                'notification' => array(
                    'title' => $title,
                    'body' => $body
                ),
                'data' => $data
            )
        );
        return $this->sendPushNotification($payload);
    }
    
    /**
     * Send notification to topic
     */
    public function sendNotificationToTopic($topic, $title, $body, $data = array()) {
        $payload = array(
            'message' => array(
                'topic' => $topic,
                'notification' => array(
                    'title' => $title,
                    'body' => $body
                ),
                'data' => $data
            )
        );
        return $this->sendPushNotification($payload);
    }
    
    /**
     * Get OAuth 2.0 access token for authentication
     */
    private function getAccessToken() {
        // Check if we have a valid token that hasn't expired
        if ($this->accessToken && time() < $this->tokenExpiry) {
            return $this->accessToken;
        }
        
        // Read service account JSON file
        if (!file_exists($this->serviceAccountPath)) {
            throw new Exception('Service account file not found: ' . $this->serviceAccountPath);
        }
        
        $serviceAccount = json_decode(file_get_contents($this->serviceAccountPath), true);
        if (!$serviceAccount) {
            throw new Exception('Invalid service account JSON file');
        }
        
        // Create JWT for authentication
        $now = time();
        $expiry = $now + 3600; // 1 hour
        
        $header = array(
            'alg' => 'RS256',
            'typ' => 'JWT'
        );
        
        $payload = array(
            'iss' => $serviceAccount['client_email'],
            'scope' => 'https://www.googleapis.com/auth/firebase.messaging',
            'aud' => 'https://oauth2.googleapis.com/token',
            'iat' => $now,
            'exp' => $expiry
        );
        
        $jwt = $this->createJWT($header, $payload, $serviceAccount['private_key']);
        
        // Exchange JWT for access token
        $tokenResponse = $this->exchangeJWTForAccessToken($jwt);
        
        if (isset($tokenResponse['access_token'])) {
            $this->accessToken = $tokenResponse['access_token'];
            $this->tokenExpiry = $now + ($tokenResponse['expires_in'] ?? 3600) - 300; // 5 min buffer
            return $this->accessToken;
        }
        
        throw new Exception('Failed to obtain access token');
    }
    
    /**
     * Create JWT token
     */
    private function createJWT($header, $payload, $privateKey) {
        $headerEncoded = $this->base64UrlEncode(json_encode($header));
        $payloadEncoded = $this->base64UrlEncode(json_encode($payload));
        
        $signature = '';
        $success = openssl_sign(
            $headerEncoded . '.' . $payloadEncoded,
            $signature,
            $privateKey,
            OPENSSL_ALGO_SHA256
        );
        
        if (!$success) {
            throw new Exception('Failed to sign JWT');
        }
        
        $signatureEncoded = $this->base64UrlEncode($signature);
        
        return $headerEncoded . '.' . $payloadEncoded . '.' . $signatureEncoded;
    }
    
    /**
     * Exchange JWT for access token
     */
    private function exchangeJWTForAccessToken($jwt) {
        $url = 'https://oauth2.googleapis.com/token';
        
        $postData = array(
            'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
            'assertion' => $jwt
        );
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, array(
            'Content-Type: application/x-www-form-urlencoded'
        ));
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        if ($response === FALSE) {
            throw new Exception('Curl failed: ' . curl_error($ch));
        }
        
        curl_close($ch);
        
        $responseData = json_decode($response, true);
        
        if ($httpCode !== 200) {
            throw new Exception('Token exchange failed: ' . ($responseData['error_description'] ?? 'Unknown error'));
        }
        
        return $responseData;
    }
    
    /**
     * Base64 URL encode
     */
    private function base64UrlEncode($data) {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }
    
    /**
     * Send push notification using FCM HTTP v1 API
     */
    private function sendPushNotification($payload) {
        // Get access token
        try {
            $accessToken = $this->getAccessToken();
        } catch (Exception $e) {
            return json_encode(array('error' => 'Authentication failed: ' . $e->getMessage()));
        }
        
        // Convert nested arrays in data to JSON strings (v1 requirement)
        if (isset($payload['message']['data'])) {
            foreach ($payload['message']['data'] as $key => $value) {
                if (is_array($value)) {
                    $payload['message']['data'][$key] = json_encode($value);
                }
            }
        }
        
        // Set up v1 API endpoint
        $url = 'https://fcm.googleapis.com/v1/projects/' . $this->projectId . '/messages:send';
        
        $headers = array(
            'Authorization: Bearer ' . $accessToken,
            'Content-Type: application/json'
        );
        
        // Open connection
        $ch = curl_init();
        
        // Set the url, number of POST vars, POST data
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true); // Enable SSL verification for security
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        
        // Execute post
        $result = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        if ($result === FALSE) {
            curl_close($ch);
            return json_encode(array('error' => 'Curl failed: ' . curl_error($ch)));
        }
        
        // Close connection
        curl_close($ch);
        
        // Parse response
        $responseData = json_decode($result, true);
        
        // Add HTTP status code to response for debugging
        $responseData['http_code'] = $httpCode;
        
        return json_encode($responseData);
    }
}
?>
