<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

$logFile = __DIR__ . "/../otp_debug.log";

// Detect calling service automatically
$callingService = "unknown";

// Lambda
if (getenv('AWS_LAMBDA_FUNCTION_NAME')) {
    $callingService = "Lambda";
}
// EC2 (check metadata endpoint)
elseif (@file_get_contents(
    'http://169.254.169.254/latest/meta-data/instance-id',
    false,
    stream_context_create(['http' => ['timeout' => 0.1]])
)) {
    $callingService = "EC2";
}
// Cognito
elseif (!empty($_SERVER['HTTP_X_AMZ_COGNITO_IDENTITY'])) {
    $callingService = "Cognito";
}

// Detect which script/API triggered this request
// Optionally, you can send a custom header "X-Calling-Script" from caller
$callingScript = $_SERVER['HTTP_X_CALLING_SCRIPT'] 
    ?? ($_SERVER['SCRIPT_FILENAME'] ?? 'unknown');

// Build log entry
$logEntry = [
    "timestamp"       => date("Y-m-d H:i:s"),
    "full_path"       => realpath(__FILE__),                        // Current script full path
    "source_ip"       => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
    "forwarded_for"   => $_SERVER['HTTP_X_FORWARDED_FOR'] ?? 'none', // Real client if behind LB
    "referer"         => $_SERVER['HTTP_REFERER'] ?? 'none',         // Caller page/script
    "user_agent"      => $_SERVER['HTTP_USER_AGENT'] ?? 'none',
    "request_method"  => $_SERVER['REQUEST_METHOD'] ?? 'none',
    "query_string"    => $_SERVER['QUERY_STRING'] ?? 'none',
    "post_data"       => $_POST ?: [],
    "calling_service" => $callingService,
    "calling_script"  => $callingScript
];

// Append log as JSON line
file_put_contents($logFile, json_encode($logEntry) . PHP_EOL, FILE_APPEND);

require_once __DIR__ . '/../aws_get_env.php'; // fetch AWS keys dynamically
require __DIR__ . '/../vendor/autoload.php';

use Aws\Sns\SnsClient;
use Aws\Exception\AwsException;

// Input validation and sanitization
$phonenumber = $_POST["username"] ?? null;
$otp         = $_POST["iveposOtp"] ?? null;

if (!$phonenumber || !$otp) {
    echo json_encode([
        "status" => "failed",
        "error"  => "Phone number and otp is missing"
    ]);
    exit;
}

// Blocked number codes
$blockedCodes = ['+880', '+964', '+20','+93'];
if (isBlockedNumber($phonenumber, $blockedCodes)) {
    echo json_encode([
        "status" => "failed",
        "error"  => "Blocked phone number"
    ]);
    exit;
}

// 🔒 Block specific phone numbers
$blockedNumbers = ['+919581497872'];  // Manager’s number
if (in_array($phonenumber, $blockedNumbers)) {
    echo json_encode([
        "status" => "failed",
        "error"  => "This number is blocked from receiving OTPs."
    ]);
    exit;
}


// Fetch AWS credentials dynamically
$awsAccessKey = get_aws_secret("AWS-ACCESS-KEY");
$awsSecretKey = get_aws_secret("AWS-SECRET-KEY");

// SNS Client
$sns = new SnsClient([
    'credentials' => [
        'key'    => $awsAccessKey,
        'secret' => $awsSecretKey
    ],
    'region'  => 'ap-south-1',
    'version' => 'latest'
]);

$message = 'Your IVEPOS/Intuition OTP for secure login is : ' . $otp;

$args = [
    'Message'     => $message,
    'PhoneNumber' => $phonenumber
];

try {
    $result = $sns->publish($args);
    echo json_encode([
        "status"      => "success",
        "otp"         => $otp,
        "phoneNumber" => $phonenumber,
        "message"     => $result
    ]);
} catch (AwsException $e) {
    error_log($e->getMessage());
    echo json_encode([
        "status" => "failed",
        "error"  => $e->getAwsErrorMessage()
    ]);
}

// Database connection
$servername = "ivepos-db-production-regular.mysql.database.azure.com";
$dbUsername = "iveposadmin";
$dbPassword = "Intuition123";
$dbname     = "ivepos_support";

// Create connection
$conn = new mysqli($servername, $dbUsername, $dbPassword, $dbname);
if ($conn->connect_error) {
    error_log("DB connection failed: " . $conn->connect_error);
    exit;
}

// Upsert OTP
$stmt = $conn->prepare("SELECT * FROM `signin_otp_log` WHERE `username`=?");
$stmt->bind_param("s", $phonenumber);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows > 0) {
    $stmt = $conn->prepare("UPDATE `signin_otp_log` SET `otp`=? WHERE `username`=?");
    $stmt->bind_param("ss", $otp, $phonenumber);
    $stmt->execute();
} else {
    $stmt = $conn->prepare("INSERT INTO `signin_otp_log` (`username`, `otp`) VALUES (?, ?)");
    $stmt->bind_param("ss", $phonenumber, $otp);
    $stmt->execute();
}

$conn->close();

// Function to check if phone number starts with any blocked code
function isBlockedNumber($phonenumber, $blockedCodes) {
    foreach ($blockedCodes as $code) {
        if (strpos($phonenumber, $code) === 0) {
            return true;
        }
    }
    return false;
}
?>
